/**
 * Color Label Helper
 * Erstellt ein Farbfeld + Textlabel (HEX, RGB, CMYK) fr die Fllfarbe
 * Untersttzt: RGBColor, CMYKColor, SpotColor
 * Autor: (optional) 
 */

function main() {
    if (app.documents.length === 0) {
        alert("Kein Dokument geffnet.");
        return;
    }

    var doc = app.activeDocument;

    if (doc.selection.length === 0) {
        alert("Bitte zuerst ein Objekt mit Fllfarbe auswhlen.");
        return;
    }

    var obj = doc.selection[0];
    if (!obj.fillColor) {
        alert("Das ausgewhlte Objekt hat keine Fllfarbe.");
        return;
    }

    var color = obj.fillColor;
    var baseColor = color;
    var rgb = {};
    var cmyk = {};
    var hex = "";

    // Hilfsfunktionen
    function padHex(n) {
        var h = n.toString(16);
        return (h.length < 2) ? "0" + h : h;
    }

    function clamp(v, min, max) {
        return Math.min(max, Math.max(min, v));
    }

    // SpotColor behandeln ? Basisfarbe extrahieren
    if (color.typename === "SpotColor") {
        baseColor = color.spot.color;
    }

    // Farbtypen auswerten (RGB oder CMYK)
    if (baseColor.typename === "RGBColor") {
        rgb.r = Math.round(baseColor.red);
        rgb.g = Math.round(baseColor.green);
        rgb.b = Math.round(baseColor.blue);

        // In CMYK umrechnen (einfaches Modell)
        var r = rgb.r / 255;
        var g = rgb.g / 255;
        var b = rgb.b / 255;

        var k = 1 - Math.max(r, g, b);
        var c = (1 - r - k) / (1 - k);
        var m = (1 - g - k) / (1 - k);
        var y = (1 - b - k) / (1 - k);

        if (isNaN(c)) c = 0;
        if (isNaN(m)) m = 0;
        if (isNaN(y)) y = 0;

        cmyk.c = Math.round(c * 100);
        cmyk.m = Math.round(m * 100);
        cmyk.y = Math.round(y * 100);
        cmyk.k = Math.round(k * 100);

    } else if (baseColor.typename === "CMYKColor") {
        cmyk.c = Math.round(baseColor.cyan);
        cmyk.m = Math.round(baseColor.magenta);
        cmyk.y = Math.round(baseColor.yellow);
        cmyk.k = Math.round(baseColor.black);

        // CMYK ? RGB
        var C = cmyk.c / 100;
        var M = cmyk.m / 100;
        var Y = cmyk.y / 100;
        var K = cmyk.k / 100;

        rgb.r = Math.round(255 * (1 - C) * (1 - K));
        rgb.g = Math.round(255 * (1 - M) * (1 - K));
        rgb.b = Math.round(255 * (1 - Y) * (1 - K));

    } else {
        alert("Dieser Farbtyp wird nicht untersttzt: " + baseColor.typename);
        return;
    }

    // Begrenzen
    rgb.r = clamp(rgb.r, 0, 255);
    rgb.g = clamp(rgb.g, 0, 255);
    rgb.b = clamp(rgb.b, 0, 255);

    // HEX
    hex = "#" + padHex(rgb.r) + padHex(rgb.g) + padHex(rgb.b);

    // Position neben dem Objekt
    var gb = obj.geometricBounds;
    var top = gb[0];
    var right = gb[3];

    var padding = 10;
    var squareSize = 20;
    var textOffset = 6;

    var group = doc.groupItems.add();

    // Farbkasten zeichnen
    var rect = group.pathItems.rectangle(top, right + padding, squareSize, squareSize);
    rect.filled = true;
    rect.fillColor = baseColor;
    rect.stroked = true;
    rect.strokeWidth = 0.25;
    rect.strokeColor = new NoColor();

    // Textblock erzeugen
    var tf = group.textFrames.add();
    tf.contents = 
        "HEX: " + hex +
        "\rRGB: " + rgb.r + " / " + rgb.g + " / " + rgb.b +
        "\rCMYK: " + cmyk.c + " / " + cmyk.m + " / " + cmyk.y + " / " + cmyk.k;

    tf.textRange.characterAttributes.size = 8;
    tf.textRange.characterAttributes.leading = 9.6;

    tf.left = right + padding + squareSize + textOffset;
    tf.top  = top - 2;

    group.left = right + padding;

    alert("Farbetikette erstellt.");
}

main();